<?php

namespace App\Http\Controllers;

use App\Models\Attendance;
use App\Models\Member;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;

class AttendanceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $type = $request->query('type');
        $date = Session::get('attendance_date');

        if (empty($date)) {
            $date = Carbon::now()->startOfDay();
        }

        if ($type == 'next') {
            $date = add_month($date);
        }

        if ($type == 'prev') {
            $date = sub_month($date);
        }

        Session::put('attendance_date', $date);

        $dates = date_range($date);
        $days = total_days($date);

        $paginate = true;

        if ($date instanceof Carbon) {
            $year = $date->year;
            $month = $date->month;
        } else {
            $date = Carbon::parse($date);

            $year = $date->year;
            $month = $date->month;
        }

        $members = Member::withAttendanceInMonth($year, $month)->orderBy('created_at','DESC')->paginate(20)->onEachSide(0);

        return view('attendance.index', compact('members', 'paginate', 'dates', 'days'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $date = Session::get('attendance_date');
        $members = $request->input('members');
        if(!empty($members)) {     
            foreach ($members as $member) {
                $member_id = $member['id'];
    
                foreach ($member['days'] as $day => $status) {
                    if (!empty($status)) {
                        $attendance_date = modify_day($date, $day);
                        // CHECK IF ATTENDANCE ALREADY EXISTS FOR THE MEMBER IN THE SELECTED DATE.
                        $attendance = Attendance::where([
                            'member_id' => $member_id,
                            'attendance_date' => $attendance_date,
                        ])->first();
    
                        if ($attendance) {
                            // UPDATE EXISTING ATTENDANCE RECORD IF FOUND.
                            $attendance->update(['status' => $status]);
                        } else {
                            // CREATE A NEW ATTENDANCE RECORD IF NOT FOUND.
                            $attendance = new Attendance([
                                'member_id' => $member_id,
                                'attendance_date' => $attendance_date,
                                'status' => $status,
                                'user_id' => Auth::user()->id,
                            ]);
                            $attendance->save();
                        }
                    }
                }
            }
            return redirect()->route('attendance.index')->with('success', 'Saved successfully');
        }
        else 
        {
            return redirect()->route('attendance.index')->with('error', 'No member selected for attendance');
        }
    }

    public function create(Request $request) {
        if ($request->has('search')) {
            $keyword = $request->input('search');
            $date = $request->input('date');

            $members = Member::with('latest_membership')->withAttendance($date)
                ->where('name', 'LIKE', '%' . $keyword . '%')
                ->orWhere('reg_no', 'LIKE', '%' . $keyword . '%')
                ->orWhere('phone', 'LIKE', '%' . $keyword . '%')
                ->orderBy('id', 'DESC')->get();

            return view('attendance.manual', compact('date', 'members'));
        } else {
            $date = Carbon::now()->toDateString();
            $members = Member::with('latest_membership')->withAttendance($date)->orderBy('id', 'DESC')->limit(15)->get();

            return view('attendance.manual', compact('date', 'members'));
        }
    }

    public function store_manual(Request $request)
    {
        $member_ids = $request->input('selected_members');
        $status = $request->input('status');
        $date = $request->input('date');

        $ids = explode(",", $member_ids);
        $ids = array_map('intval', $ids);

        foreach ($ids as $id) {
            if (!empty($status)) {
                // CHECK IF ATTENDANCE ALREADY EXISTS FOR THE MEMBER IN THE SELECTED DATE.
                $attendance = Attendance::where([
                    'member_id' => $id,
                    'attendance_date' => $date,
                ])->first();

                if ($attendance) {
                    // UPDATE EXISTING ATTENDANCE RECORD IF FOUND.
                    $attendance->update(['status' => $status]);
                } else {
                    // CREATE A NEW ATTENDANCE RECORD IF NOT FOUND.
                    $attendance = new Attendance([
                        'member_id' => $id,
                        'attendance_date' => $date,
                        'status' => $status,
                        'user_id' => Auth::user()->id,
                    ]);
                    $attendance->save();
                }
            }
        }

        return redirect()->route('attendance.create')->with('success', 'Attendance Taken Successfully');
    }
}